library IEEE;			
use IEEE.STD_LOGIC_1164.ALL;
use IEEE.numeric_std.ALL;

library unisim;
use unisim.vcomponents.all;

use work.IOB_Config.ALL;
use work.VT52_cfg.ALL;

entity VT52 is
	generic
		(print_addr: DevID := O"03";
		 keyb_addr: DevID := O"04");
	port
		( -- clocks
		 clk: in std_logic;
		 clkdiv24: in std_logic;
		 vidclk: in std_logic;

		 -- CPU bus interface
		 reset: in boolean;
		 IOTact: in boolean;
		 IOTdev: in DevID;
		 IOTcmd: in DevCmd;

		 clk_write_n : in std_logic;
		 clk_lxdar_n : in std_logic;
		 IOTread : in boolean;
		 dx : inout std_logic_vector(0 to 11);
		 cpu_c0_n : out std_logic;
		 cpu_c1_n : out std_logic;
		 cpu_skip_n : out std_logic;
		 IRQ: out std_logic;

		 -- Keyboard
		 kb_clk: inout std_logic;
		 kb_data: inout std_logic;

		 -- VGA
		 vga_RGB: out RGB;
		 vga_HS: out std_logic;
		 vga_VS: out std_logic);
end VT52;

architecture RTL of VT52 is

	constant BELLDURATION: integer := 15;	-- frames the bell lasts - ~1/4 sec
	constant INTERCHDELAY: integer := 767;	-- clocks delay between characters = ~ 26 us

	component ps2kbd is
		port
			(reset: in boolean;
			 clk: in std_logic;
			 clkdiv24: in std_logic;
		  	 alt_keypad_mode: in boolean;
			 hold_screen_mode: in boolean;
			 hold_line_release_tgl: out std_logic;
			 hold_scrn_release_tgl: out std_logic;
          ack : in std_logic;
          data : out std_logic_vector(7 downto 0);
          data_avail : out std_logic;
          kb_clk_raw : inout std_logic;
          kb_data_raw : inout std_logic);
	end component;

	component VideoGen is
		generic
			(CLKFREQ: integer := 29491200;
			 HCENTER: real := 0.50;
			 VCENTER: real := 0.50;
			 curs_blink: boolean := true;
			 curs_block: boolean := false;
	       fore_color: RGB := "111";
	       back_color: RGB := "000");
	    port
		 	(reset: in boolean;
	       clk : in std_logic;
	       X : out COLUMN;
	       Y : out ROW;
			 text_rd: out std_logic;
	       text : in std_logic_vector(7 downto 0);
	       curs_X : in COLUMN;
	       curs_Y : in ROW;
			 invert_screen: in boolean;
	       vga_RGB : out RGB;
	       vga_HSYNC : out std_logic;
	       vga_VSYNC : out std_logic);
	end component;

	component addrmap
		port
			(x: in COLUMN;
			 y: in ROW;
			 ybase: in ROW;
			 addr: OUT std_logic_vector(10 downto 0));
	end component;

	component PosEdge
		port
		  (reset: in boolean;
			clk: in std_logic;
			inp: in std_logic;          
			outp: out std_logic);
	end component;

	attribute INIT_00: string;  attribute INIT_01: string; attribute INIT_02: string;  attribute INIT_03: string;
	attribute INIT_04: string;  attribute INIT_05: string; attribute INIT_06: string;  attribute INIT_07: string;
	attribute INIT_08: string;  attribute INIT_09: string; attribute INIT_0A: string;  attribute INIT_0B: string;
	attribute INIT_0C: string;  attribute INIT_0D: string; attribute INIT_0E: string;  attribute INIT_0F: string;

	-- flags and interrupts
	signal IE, printer_flag, keyb_flag: std_logic;

	signal write: boolean;

	-- keyboard logic

	subtype Byte is std_logic_vector(7 downto 0);
	signal rsel, advance, alt_keypad_mode: boolean;
	signal rselread_n, rsr_ne, kb_avail, kb_ack: std_logic;
	signal kb_ASCII: Byte;
	type FIFO16 is array(0 to 15) of Byte;
	signal keyQ: FIFO16;
	signal head, tail: integer range 0 to 15;
	signal inppacing: integer range 0 to INTERCHDELAY/24;
	signal inpready, inpavail: std_logic;
	signal iclrflag: boolean;

	-- video logic
	signal vga_VS_int, VS_pe: std_logic;

	signal scan_X, curs_X, save_X: COLUMN;
	signal top_Y, scan_Y, curs_Y, save_Y: ROW;
	signal invtext, scr_rd_en, scr_wr_en: std_logic;
	signal scr_rd_addr, scr_wr_addr: std_logic_vector(10 downto 0);
	signal scr_rd, scr_wr: std_logic_vector(7 downto 0);

	-- output control signals
	signal outavail, longop, THRE, THRE_1, THRE_2: std_logic;
	signal oclrflag, osetflag: boolean;
	signal outpacing: integer range 0 to INTERCHDELAY/24;

	signal tsel: boolean;
	signal outc: std_logic_vector(7 downto 0);
	signal bell: boolean;
	signal bell_count: integer range 0 to BELLDURATION;
	signal identify: integer range 0 to 3;

	type OutState is
		(
		Idle,
		IncrX,
		WaitCmd,
		WaitX,
		WaitY,
		ClearLine,
		WriteEOL,
		WriteEOS
		);

	signal ostate: OutState;

attribute clock_signal: string;
attribute clock_signal of outavail : signal is "yes";
attribute clock_signal of inpavail : signal is "yes";

begin

	-- misc

	write <= (clk_write_n = '0');
	IRQ <= IE and (printer_flag or keyb_flag);

	-- Input (PS/2 keyboard)

	keyb: ps2kbd port map
		(reset => reset, clk => clk, clkdiv24 => clkdiv24,
		 alt_keypad_mode => alt_keypad_mode,
		 hold_screen_mode => false,
		 hold_line_release_tgl => open,
		 hold_scrn_release_tgl => open,
		 ack => kb_ack, data => kb_ASCII, data_avail => kb_avail,
		 kb_clk_raw => kb_clk, kb_data_raw => kb_data);

	-- FIFO filling process
	process (reset, clk, identify, tail, kb_avail, kb_ASCII)
	begin
		if reset then
			tail <= 0;
			kb_ack <= '0';
		elsif rising_edge(clk) then
			if kb_ack = '1' then
				kb_ack <= '0';
			elsif kb_avail = '1' then
				keyQ(tail) <= kb_ASCII;
				tail <= (tail + 1) mod keyQ'length;
				kb_ack <= '1';				
			elsif identify = 1 then
				keyQ(tail) <= "01001011";	-- 'K'
				tail <= (tail + 1) mod keyQ'length;
			elsif identify = 2 then
				keyQ(tail) <= "00101111";	-- '/'
				tail <= (tail + 1) mod keyQ'length;
			elsif identify = 2 then
				keyQ(tail) <= "00011011";	-- 'Esc'
				tail <= (tail + 1) mod keyQ'length;
			end if;
		end if;
	end process;

	-- manage input pacing timer
	process (reset, clkdiv24, advance, inppacing)
	begin
		if reset then
			inppacing <= 0;
		elsif advance then
			inppacing <= INTERCHDELAY/24;
		elsif rising_edge(clkdiv24) then
			if inppacing /= 0 then
				inppacing <= inppacing - 1;
			end if;
		end if;
	end process;

	-- manage keyboard flag
	inpready <= '1' when head /= tail else '0';
	inpavail <= '1' when (inpready = '1') and (inppacing = 0) else '0';

	-- reading process
	rsel <= IOTact and (IOTdev = keyb_addr);
	rselread_n <= '0' when rsel and IOTread else '1';

	dx(4 to 11) <= keyQ(head)(7 downto 0) when rselread_n = '0' else (others => 'Z');
	dx(0 to 3) <= (others => '0') when rselread_n = '0' else (others => 'Z');

	-- keyboard flag management
	iclrflag <= rsel and write and ((IOTcmd = KCF) or (IOTcmd = KRB));
				
	process (reset, inpavail, iclrflag)
	begin
		if reset or iclrflag then
			keyb_flag <= '0';
		elsif rising_edge(inpavail) then
			keyb_flag <= '1';
		end if;
	end process;

	-- need to advance head after IOT that reads char
	process (clk_write_n, rsel, IOTcmd)
	begin
		if rising_edge(clk_write_n) then
			advance <= rsel and (IOTcmd = KRB);
		end if;
	end process;

	process (reset, clk_lxdar_n, advance, head)
	begin
		if reset then
			head <= 0;
		elsif rising_edge(clk_lxdar_n) then
			if advance then
				head <= (head + 1) mod keyQ'length;
			end if;
		end if;
	end process;

	-- latch IE bit
	process (reset, clk_write_n)
	begin
		if reset then
			IE <= '1';
		elsif rising_edge(clk_write_n) then
			if rsel and (IOTcmd = KIE) then
				IE <= dx(11);
			end if;
		end if;
	end process;

	-- c0/1/skip feedback
	process (rsel, write, iotcmd, keyb_flag)
	begin
		if rsel and write then
		-- n.b. clk_write_n is synchronized, therefore delayed, so
		-- TRH should be met

			-- c0/c1
			case IOTcmd is

				when KCC | KRB =>
					cpu_c0_n <= '0';

				when others =>
					cpu_c0_n <= 'Z';

			end case;

			case IOTcmd is

				when KRS | KRB =>
					cpu_c1_n <= '0';

				when others =>
					cpu_c1_n <= 'Z';

			end case;

			-- skip
			case IOTcmd is

				when KSF =>
					if keyb_flag = '1' then
						cpu_skip_n <= '0';
					else
						cpu_skip_n <= 'Z';
					end if;

				when others =>
					cpu_skip_n <= 'Z';

			end case;
		else
			cpu_c0_n <= 'Z';
			cpu_c1_n <= 'Z';
			cpu_skip_n <= 'Z';
		end if;
	end process;

	-- Output (VGA display)

	tsel <= IOTact and (IOTdev = print_addr);

	-- format the port data into standard 7-bit ASCII
	outc <= '0' & dx(5) & dx(6) & dx(7) & dx(8) & dx(9) & dx(10) & dx(11);

	THRE <= '1' when THRE_1 = THRE_2 else '0';

	-- manage output pacing timer
	process (reset, clkdiv24, THRE, outpacing)
	begin
		if reset then
			outpacing <= 0;
		elsif THRE = '0' then
			outpacing <= INTERCHDELAY/24;
		elsif rising_edge(clkdiv24) then
			if outpacing /= 0 then
				outpacing <= outpacing - 1;
			end if;
		end if;
	end process;

	outavail <= '1' when (THRE = '1') and (longop = '0') and (outpacing = 0) else '0';

	-- manage print flag
	osetflag <= tsel and write and (IOTcmd = TFL);
	oclrflag <= tsel and write and ((IOTcmd = TCF) or (IOTcmd = TLS));
				
	process (reset, outavail, osetflag, oclrflag)
	begin
		if reset or oclrflag then
			printer_flag <= '0';
		elsif osetflag then
			printer_flag <= '1';
		elsif rising_edge(outavail) then
			printer_flag <= '1';
		end if;
	end process;

	-- manage transmit
	process (reset, clk_write_n, tsel, THRE_2)
	begin
		if reset then
			THRE_1 <= '0';
		elsif rising_edge(clk_write_n) then
			if tsel and ((IOTcmd = TPC) or (IOTcmd = TLS)) then
				THRE_1 <= not THRE_2;
			end if;
		end if;
	end process;

	-- c0/1/skip feedback
	process (tsel, write, iotcmd, printer_flag, keyb_flag)
	begin
		if tsel and write then
		-- n.b. clk_write_n is synchronized, therefore delayed, so
		-- TRH should be met

			-- skip
			case IOTcmd is

				when TSF =>
					if printer_flag = '1' then
						cpu_skip_n <= '0';
					else
						cpu_skip_n <= 'Z';
					end if;

				when TSK =>
					if (printer_flag or keyb_flag) = '1' then
						cpu_skip_n <= '0';
					else
						cpu_skip_n <= 'Z';
					end if;

				when others =>
					cpu_skip_n <= 'Z';

			end case;
		else
			cpu_skip_n <= 'Z';
		end if;
	end process;


	-- map the user cursor and screen refresh cursor to RAM addresses
	curs_map: addrmap
		port map
			(x => curs_X, Y => curs_Y, Ybase => top_Y, addr => scr_wr_addr);

	scan_map: addrmap
		port map
			(x => scan_X, Y => scan_Y, Ybase => top_Y, addr => scr_rd_addr);

	-- the text buffer
	textram: for n in 0 to 3 generate
	begin
		textb: RAMB4_S2_S2
			port map
				(
				-- reading port
				RSTA => '0', WEA => '0', ENA => scr_rd_en, CLKA => clk,
				ADDRA => scr_rd_addr, DIA => "00",
				DOA => scr_rd(n*2+1 downto n*2),

				-- writing port
				RSTB => '0', WEB => '1', ENB => scr_wr_en, CLKB => clk,
				ADDRB => scr_wr_addr, DOB => open,
				DIB => scr_wr(n*2+1 downto n*2)
				);
	end generate;

	vga_VS <= vga_VS_int;

	-- this module generates the timing and video signals, and converts the ASCII into
	-- its corresponding scan patterns
	video: VideoGen
		generic map
			(curs_blink => CURSBLINK,
			 curs_block => CURSBLOCK,
	       fore_color => FOREG,
	       back_color => BACKG)
	    port map
		 	(reset => reset, clk => vidclk,
			 X => scan_x, Y => scan_y,
	       text_rd => scr_rd_en, text => scr_rd,
	       curs_X => curs_X, curs_Y => curs_Y,
			 invert_screen => bell,
	       vga_RGB => vga_RGB, vga_HSYNC => vga_HS, vga_VSYNC => vga_VS_int);

	-- character writing logic.
	-- this is the entire state machine that interprets character writes as
	-- displayable characters or executes control or escape functions

	vs_det: PosEdge port map (reset, clk, vga_VS_int, VS_pe);

	-- the state machine
	process (reset, clk, ostate)
	begin
		if reset then

			THRE_2 <= '0';
			longop <= '0';

			top_Y <= 0;
			curs_X <= 0;
			curs_Y <= 0;
			save_X <= 0;
			save_Y <= 0;
			scr_wr <= "00100000";	-- space
			scr_wr_en <= '1';
			ostate <= WriteEOS;

			bell_count <= 0;
			invtext <= '0';
			alt_keypad_mode <= false;

		elsif rising_edge(clk) then

			-- if identify sequence in progress, do next
			if identify /= 0 then
				identify <= identify - 1;
			end if;

			case ostate is

				when Idle =>

					-- reset various state flags
					identify <= 0;
					longop <= '0';

					-- bell counter
					if VS_PE = '1' then
						if bell_count /= 0 then
							bell_count <= bell_count - 1;
						else
							bell <= false;
						end if;
					end if;

					if THRE = '0' then

						THRE_2 <= THRE_1;	-- handshake character

						if outc(7 downto 5) /= "000" then

							-- write the character to the screen
							scr_wr <= invtext & outc(6 downto 0);
							scr_wr_en <= '1';
							ostate <= IncrX;

						else

							-- interpret control character
							case outc(4 downto 0) is
								when "00111" =>	-- BEL
									bell_count <= BELLDURATION;
									bell <= true;

								when "01000" =>	-- BS
									if curs_X /= 0 then
										curs_X <= curs_X - 1;
									end if;

								when "01001" =>	-- Tab
									if curs_X < 72 then
										curs_X <= to_integer(to_unsigned(curs_X, 7) or "111") + 1;
									end if;

								when "01010" =>	-- LF
									if curs_Y = ROWS-1 then
										if top_Y = ROWS-1 then
											top_Y <= 0;
										else
											top_Y <= top_Y + 1;
										end if;
										ostate <= ClearLine;
									else
										curs_Y <= curs_Y + 1;
									end if;

								when "01101" =>	-- CR
									curs_X <= 0;

								when "01110" =>	-- SO	(inverse)
									invtext <= '1';

								when "01111" =>	-- SI	(normal)
									invtext <= '0';

								when "11011" =>	-- ESC
									ostate <= WaitCmd;

								when others =>
									null;

							end case;

						end if;
					end if;

				when WaitCmd =>

					if THRE = '0' then

						THRE_2 <= THRE_1;	-- handshake character

						case outc is

							when "01000001" =>	-- 'A'	cursor up
								if curs_Y /= 0 then
									curs_Y <= curs_Y - 1;
								end if;
								ostate <= Idle;
							
							when "01000010" =>	-- 'B'	cursor down
								if curs_Y /= ROWS-1 then
									curs_Y <= curs_Y + 1;
								end if;
								ostate <= Idle;
							
							when "01000011" =>	-- 'C'	cursor right
								if curs_X /= COLUMNS-1 then
									curs_X <= curs_X + 1;
								end if;
								ostate <= Idle;
							
							when "01000100" =>	-- 'D'	cursor left
								if curs_X /= 0 then
									curs_X <= curs_X - 1;
								end if;
								ostate <= Idle;

							when "01001000" =>	-- 'H'	cursor home
								curs_X <= 0;
								curs_Y <= 0;
								ostate <= Idle;


							when "00111101" =>	-- '='	alt keypad mode
								alt_keypad_mode <= true;
								ostate <= Idle;

							when "00111110" =>	-- '>'	norm keypad mode
								alt_keypad_mode <= false;
								ostate <= Idle;

							when "01001001" =>	-- 'I'	reverse linefeed
								if curs_Y = 0 then
									if top_Y = 0 then
										top_Y <= ROWS-1;
									else
										top_Y <= top_Y - 1;
									end if;
									ostate <= ClearLine;
								else
									curs_Y <= curs_Y - 1;
									ostate <= Idle;
								end if;

							when "01001010" =>	-- 'J'	erase to end of screen
								longop <= '1';
								save_X <= curs_X;
								save_Y <= curs_Y;
								scr_wr <= "00100000";	-- space
								scr_wr_en <= '1';
								ostate <= WriteEOS;

							when "01001011" =>	-- 'K'	erase to end of line
								longop <= '1';
								save_X <= curs_X;
								scr_wr <= "00100000";	-- space
								scr_wr_en <= '1';
								ostate <= WriteEOL;

							when "01011010" =>	-- 'Z'	identify
								longop <= '1';
								identify <= 3;
								ostate <= Idle;
							
							when "01011001" =>	-- 'Y'	cursor positioning
								ostate <= WaitY;

							when others =>
								ostate <= Idle;

						end case;

					end if;

				when WaitY =>

					if THRE = '0' then
						THRE_2 <= THRE_1;	-- handshake character
						curs_Y <= to_integer(unsigned(outc)) - 32;
						ostate <= WaitX;
					end if;

				when WaitX =>

					if THRE = '0' then
						THRE_2 <= THRE_1;	-- handshake character
						curs_X <= to_integer(unsigned(outc)) - 32;
						ostate <= Idle;
					end if;

				when IncrX =>
					scr_wr_en <= '0';
					if curs_X /= COLUMNS-1 then
						curs_X <= curs_X + 1;
					end if;
					ostate <= Idle;

				when ClearLine =>				-- common code for fwd and rev line feed
					longop <= '1';
					save_X <= curs_X;
					curs_X <= 0;
					scr_wr <= "00100000";	-- space
					scr_wr_en <= '1';
					ostate <= WriteEOL;

				when WriteEOL =>				-- keep writing until end of line
					if curs_X = COLUMNS-1 then
						scr_wr_en <= '0';
						ostate <= Idle;
						curs_X <= save_X;
					else
						curs_X <= curs_X + 1;
					end if;

				when WriteEOS =>				-- keep writing until end of screen
					if curs_X = COLUMNS-1 then
						if curs_Y = ROWS-1 then
							scr_wr_en <= '0';
							ostate <= Idle;
							curs_X <= save_X;
							curs_Y <= save_Y;
						else
							curs_Y <= curs_Y + 1;
							curs_X <= 0;
						end if;
					else
						curs_X <= curs_X + 1;
					end if;

				end case;

		end if;
	end process;

end RTL;
