library IEEE;
use IEEE.STD_LOGIC_1164.ALL;
use IEEE.numeric_std.ALL;

library unisim;
use unisim.vcomponents.all;

use work.IOB_Config.ALL;
use work.VT52_cfg.ALL;

entity iob is
	port
	(clk_in : in std_logic;	-- 29.4912 MHz

	-- CPU bus interface
	cpu_ioclr_n : in std_logic;
	dx : inout std_logic_vector(0 to 11);
	clk_write_n_raw : in std_logic;			-- async
	cpu_read_n : in std_logic;
	cpu_sr_read_n : in std_logic;
	cpu_sr_write_n : in std_logic;
	clk_lxdar_n_raw : in std_logic;			-- async
	cpu_c0_n : out std_logic;
	cpu_c1_n : out std_logic;
	cpu_skip_n : out std_logic;
	cpu_intreq_n : out std_logic;
	cpu_intgrnt_n : inout std_logic;
	cpreq_n: inout std_logic;

	-- serial ports
	txd : out std_logic_vector(0 to NUMUARTS-1);
	rxd : in std_logic_vector(0 to NUMUARTS-1);

	-- LPT port
	lpt_ack: in std_logic;
	lpt_busy_n: in std_logic;
	lpt_paper_end_n: in std_logic;
	lpt_select_in_n: in std_logic;
	lpt_error: in std_logic;
	lpt_strobe: out std_logic;
	lpt_ddir: out std_logic;
	lpt_data: inout std_logic_vector(7 downto 0);
	lpt_init: out std_logic;
		
	-- Keyboard
	kb_clk: inout std_logic;
	kb_data: inout std_logic;

	-- VGA
	vga_RGB: out RGB;
	vga_HS: out std_logic;
	vga_VS: out std_logic;

	-- speaker
	spkr: out std_logic;

	-- real-time clock
	rtc_sclk: out std_logic;
	rtc_io: inout std_logic;
	rtc_rst: out std_logic;

	-- bits for custom applications
	iobits: inout std_logic_vector(0 to 35);

	-- special purpose
	reprogram: inout std_logic);	-- drive low to restart FPGA configuration

-- clock frequency and pin assignments, Xilinx-specific.
-- put here rather than UCF so they don't get lost
-- don't allow use of config pins for I/O by assigning to dummy signals

---- input clock
attribute period: string;
attribute period of clk_in : signal is "29.4912 MHz";

---- assign pins to match PCB
attribute loc: string;
attribute clock_buffer : string; 
attribute pullup : string;

attribute loc of clk_in : signal is "P88";
attribute loc of clk_lxdar_n_raw : signal is "P18";
attribute clock_buffer of clk_lxdar_n_raw : signal is "ibufg";
attribute loc of clk_write_n_raw : signal is "P15";
attribute clock_buffer of clk_write_n_raw : signal is "ibufg";
attribute loc of cpu_c0_n : signal is "P79";
attribute loc of cpu_c1_n : signal is "P80";
attribute loc of cpu_intreq_n : signal is "P96";
attribute loc of cpu_ioclr_n : signal is "P87";
attribute loc of cpu_read_n : signal is "P78";
attribute loc of cpu_skip_n : signal is "P76";
attribute loc of cpu_sr_read_n : signal is "P77";
attribute loc of cpu_sr_write_n : signal is "P101";
attribute loc of dx : signal is "P131 P129 P132 P122 P67 P62 P60 P57 P49 " &
									"P46 P44 P39";
attribute loc of kb_clk : signal is "P138";
attribute pullup of kb_clk : signal is "yes";
attribute loc of kb_data : signal is "P133";
attribute pullup of kb_data : signal is "yes";
attribute loc of lpt_ack : signal is "P91";
attribute loc of lpt_busy_n : signal is "P121";
attribute loc of lpt_data : signal is "P38 P47 P43 P51 P56 P59 P63 P66";
attribute loc of lpt_ddir : signal is "P28";
attribute loc of lpt_error : signal is "P137";
attribute loc of lpt_init : signal is "P7";
attribute loc of lpt_paper_end_n : signal is "P136";
attribute loc of lpt_select_in_n : signal is "P124";
attribute loc of lpt_strobe : signal is "P10";
attribute loc of reprogram : signal is "P85";
attribute loc of rxd : signal is "P23 P26 P27";
attribute loc of txd : signal is "P11 P21 P115";
attribute loc of vga_hs : signal is "P117";
attribute loc of vga_rgb : signal is "P99 P126 P120";
attribute loc of vga_vs : signal is "P102";
attribute loc of cpu_intgrnt_n : signal is "P141";
attribute loc of cpreq_n : signal is "P113";
attribute loc of iobits : signal is "P12 P5 P3 P20 P103 P93 P100 P94 P19 " &
											"P13 P4 P6 P134 P114 P139 P116 P74 P64 " &
											"P42 P84 P83 P40 P48 P65 P130 P123 P140 " &
											"P118 P75 P41 P58 P86 P22 P29 P50 P54";
attribute loc of spkr : signal is "P95";
attribute loc of rtc_rst : signal is "P112";
attribute loc of rtc_sclk : signal is "P68";
attribute loc of rtc_io : signal is "P30";

end iob;

architecture RTL of iob is

	-- PosEdge: in this implementation, most processes are
	-- synchronous to clk, so this component is used to sample
	-- slower signals, and generates a 1-clock pulse on their rising edge
	component PosEdge
		port
		  (reset: in boolean;
			clk: in std_logic;
			inp: in std_logic;          
			outp: out std_logic);
	end component;

	component KL8JA is
		generic
			(print_addr: DevID;
			keyb_addr: DevID);
		port
			( -- clocks
			clk: in std_logic;
			brg: in unsigned(7 downto 0);

			-- CPU bus interface
			reset : in boolean;
			IOTact: in boolean;
			IOTdev: in DevID;
			IOTcmd: in DevCmd;
			dx : inout std_logic_vector(0 to 11);
			clk_write_n : in std_logic;
			clk_lxdar_n : in std_logic;
			IOTread : in boolean;
			cpu_c0_n : out std_logic;
			cpu_c1_n : out std_logic;
			cpu_skip_n : out std_logic;

			-- communication
			IRQ: out std_logic;

			-- serial port
			txd : out std_logic;
			rxd : in std_logic
			);
	end component;

	component LC8E is
		generic
			(print_addr: DevID;
			 keyb_addr: DevID);
		port
			(reset: in boolean;							-- true when reset (IOCLR)
			 IOTact: in boolean;							-- true when in IOT (LXDAR)
			 IOTdev: in DevID;							-- latched IOT device code
			 IOTcmd: in DevCmd;							-- latched IOT command

			 clk_write_n : in std_logic;
			 dx : in std_logic_vector(7 downto 0);
			 cpu_skip_n : out std_logic;
			 pIRQ: out std_logic;

			 -- LPT port
			 lpt_ack: in std_logic;
			 lpt_busy_n: in std_logic;
			 lpt_paper_end_n: in std_logic;
			 lpt_select_in_n: in std_logic;
			 lpt_error: in std_logic;
			 lpt_strobe: buffer std_logic;
			 lpt_ddir: out std_logic;
			 lpt_data: inout std_logic_vector(7 downto 0);
			 lpt_init: out std_logic);
	end component;

	component VT52 is
		generic
			(print_addr: DevID;
			 keyb_addr: DevID);
		port
			( -- clocks
			clk : in std_logic;
			clkdiv24: in std_logic;
			vidclk: in std_logic;

			-- CPU bus interface
			reset : in boolean;
			IOTact: in boolean;
			IOTdev: in DevID;
			IOTcmd: in DevCmd;

			clk_write_n : in std_logic;
		 	clk_lxdar_n : in std_logic;
			IOTread : in boolean;
			dx : inout std_logic_vector(0 to 11);
			cpu_c0_n : out std_logic;
			cpu_c1_n : out std_logic;
			cpu_skip_n : out std_logic;
			IRQ: out std_logic;

			-- Keyboard
			kb_clk: inout std_logic;
			kb_data: inout std_logic;

			-- VGA
			vga_RGB: out std_logic_vector(0 to 2);
			vga_HS: out std_logic;
			vga_VS: out std_logic);
	end component;

	component Div3 is
	    port
		 	(I: in std_logic;
			 O: out std_logic);
	end component;

-- registers & signals shared between all devices

signal cpu_ioclr: std_logic;
signal reset: boolean;							-- true when reset (IOCLR)
signal IOTact: boolean;							-- true when in IOT (LXDAR)
signal IOTdev: DevID;							-- latched IOT device code
signal IOTcmd: DevCmd;							-- latched IOT command
signal IOTread: boolean;						-- true during IOT read cycle

signal dx_byte: std_logic_vector(7 downto 0); -- dx "right way round"

signal vidclk, clk, clkdiv24, clkfb, clk2x: std_logic;
signal prescale: integer range 0 to 23;
signal brg: unsigned(7 downto 0);			-- baud rate generator

attribute clock_signal: string;
attribute clock_signal of clkdiv24 : signal is "yes";
attribute clock_signal of BRG : signal is "yes";
attribute clock_signal of vidclk : signal is "yes";

signal reprogram_int: boolean;

signal clk_lxdar_n_meta, clk_write_n_meta,
		 clk_lxdar_n, clk_write_n: std_logic;

signal uIRQ: unsigned(0 to NUMUARTS-1);	-- masked flags from the serial ports
signal pIRQ: std_logic;							-- masked flag from the printer port
signal vIRQ: std_logic;							-- masked flag from VT52

-- dummy switch register registers & signals 

signal switchreg: std_logic_vector(0 to 11);
signal sr_write: std_logic;

begin

	spkr <= '1';
	rtc_sclk <= '1';
	rtc_io <= '1';
	rtc_rst <= '1';

-----------------------------------------------------------------------------------------
-- utility and shared logic
-----------------------------------------------------------------------------------------

cpu_c0_n <= 'Z';
cpu_c1_n <= 'Z';
cpu_intreq_n <= 'Z';
cpu_skip_n <= 'Z';
cpreq_n <= 'Z';
cpu_intgrnt_n <= 'Z';
dx <= (others => 'Z');

dx_byte(7 downto 0) <= dx(4 to 11);

-- synchronize important async signals

process (reset, clk2x, clk_lxdar_n_raw, clk_write_n_raw)
begin
	if reset then
		clk_lxdar_n <= '1';
		clk_write_n <= '1';
		clk_lxdar_n_meta <= '1';
		clk_write_n_meta <= '1';
	elsif rising_edge(clk2x) then
		clk_lxdar_n_meta <= clk_lxdar_n_raw;
		clk_write_n_meta <= clk_write_n_raw;
		clk_lxdar_n <= clk_lxdar_n_meta;
		clk_write_n <= clk_write_n_meta;
	end if;		
end process;

-- common IOT decoding

cpu_ioclr <= not cpu_ioclr_n;
reset <= cpu_ioclr_n = '0';

-- latch IOT device and command at start of LXDAR
process (clk_lxdar_n)
begin
	if falling_edge(clk_lxdar_n) then
		IOTdev <= unsigned(dx(3 to 8));
		IOTcmd <= unsigned(dx(9 to 11));
	end if;
end process;

IOTact <= (clk_lxdar_n = '0');

-- here we stretch the read signal from when it is active
-- until the end of LXDAR; IM6120 wants IOT read data
-- at end of LXDAR, not READ
process (reset, clk, IOTact, cpu_read_n)
begin
	if reset or not IOTact then
		IOTread <= false;
	elsif rising_edge(clk) then
		if cpu_read_n = '0' then
			IOTread <= true;
		end if;
	end if;
end process;

--- special IOT to initiate FPGA reprogram

reprogram <= '0' when reprogram_int else 'Z';

process (clk_write_n, dx)
begin
	if rising_edge(clk_write_n) then
		if IOTact and (IOTdev = SPCBASE) and
		   (IOTcmd = SPCREPROG) and
			(dx = O"1234") then
			reprogram_int <= true;			
		else
			reprogram_int <= false;
		end if;
	end if;
end process;

---- bring CPU IRQ low when printer or serial ports request it
cpu_intreq_n <= '0' when pIRQ = '1' or vIRQ = '1' or (uIRQ /= 0) else 'Z';

-- baud rate generator for uarts

-- to simplify the circuit, we use one oscillator for both
-- the video and the serial clocks.  The frequency chosen
-- is one that's close to the required video dot clock and
-- also dividable down to max_baud_rate*16 with little or no
-- error (2% is the max we want).  As it happens, the fairly
-- standard 29.4912MHz oscillator meets both these criteria
-- pretty well.
--
-- the second wrinkle is that we use a special Xilinx feature to double
-- the clock rate.  This clock is used to synchronize the external
-- asynchronous signals, reducing the delay of the two-clock lag.
-- This is done by instantiating some library objects that
-- represent clock DLL features built into the FPGA.

clk1: clkdll port map (RST => cpu_ioclr, CLKIN => clk_in,
							  CLK0 => clk, CLKFB => clkfb, CLK2X => clk2x);

fbnet: BUFG port map (I => clk, O => clkfb);

-- video gets oscillator clock directly
vidclk <= clk;

-- The baud rate clock is 38400*16, which is the input clock / 24.
-- we then divide this by powers of two for the various baud rates;
-- the UARTs tap these dividers by indexing into the brg() array.
-- the /24 clock is also exposed for places that a slower clock
-- is useful, like delay counters.
process (reset, clk)
begin
	if reset then
		brg <= (others => '0');
		prescale <= 0;
		clkdiv24 <= '0';
	elsif rising_edge(clk) then
		if prescale = 23 then
			prescale <= 0;
			clkdiv24 <= '1';
			brg <= brg + 1;
		else
			prescale <= prescale + 1;
			clkdiv24 <= '0';
		end if;
	end if;
end process;

-----------------------------------------------------------------------------------------
-- switch register dummy implementation - just a latch
-----------------------------------------------------------------------------------------

process (cpu_sr_read_n, dx, switchreg)
begin
	if cpu_sr_read_n = '0' then
		dx <= switchreg;
	else
		dx <= (others => 'Z');
	end if;
end process;

sr_w_detect: PosEdge port map (reset, clk, cpu_sr_write_n, sr_write);

process (clk, sr_write)
begin
	if rising_edge(clk) then
		if sr_write = '1' then
			switchreg <= dx;
		end if;
	end if;
end process;

-----------------------------------------------------------------------------------------
-- KL8JA serial ports
-----------------------------------------------------------------------------------------

KL8JAs: for idx in 0 to NUMUARTS-1 generate
begin

	sp: KL8JA
		generic map
		(
		print_addr => UARTOBASE(idx),
		keyb_addr => UARTIBASE(idx)
		)
		port map
			(clk => clk, reset => reset,
			 brg => brg,
			 IOTact => IOTact, IOTdev => IOTdev, IOTread => IOTread,
			 IOTcmd => IOTcmd, dx => dx,
			 clk_write_n => clk_write_n, clk_lxdar_n => clk_lxdar_n,
			 cpu_c0_n => cpu_c0_n, cpu_c1_n => cpu_c1_n, cpu_skip_n => cpu_skip_n,
			 IRQ => uIRQ(idx), txd => txd(idx), rxd => rxd(idx));

end generate;

-----------------------------------------------------------------------------------------
-- LC8E printer interface with Centronics output
-----------------------------------------------------------------------------------------

parpt_if: if PARPTIBASE /= O"00" generate
signal lpt_strobe_i: std_logic;
begin

lpt_strobe <= lpt_strobe_i;

pp: LC8E
	generic map
		(print_addr => PARPTOBASE, keyb_addr => PARPTIBASE)
	port map
		(reset => reset, IOTact => IOTact, clk_write_n => clk_write_n,
		 IOTdev => IOTdev, IOTcmd => IOTcmd,
		 dx => dx_byte,
		 cpu_skip_n => cpu_skip_n, pIRQ => pIRQ,
		 lpt_ack => lpt_ack, lpt_busy_n => lpt_busy_n,
		 lpt_paper_end_n => lpt_paper_end_n,
		 lpt_select_in_n => lpt_select_in_n, lpt_error => lpt_error,
		 lpt_strobe => lpt_strobe_i, lpt_ddir => lpt_ddir,
		 lpt_data => lpt_data, lpt_init => lpt_init);

end generate;

-----------------------------------------------------------------------------------------
-- VT52 via KL8E interface
-----------------------------------------------------------------------------------------

vt: VT52
	generic map
		(print_addr => VT52OBASE, keyb_addr => VT52IBASE)
	port map
		(reset => reset, clk => clk, clkdiv24 => clkdiv24, vidclk => vidclk,
		 IOTact => IOTact, IOTdev => IOTdev,	IOTcmd => IOTcmd,
		 clk_write_n => clk_write_n, clk_lxdar_n => clk_lxdar_n,
		 IOTread => IOTread, dx => dx,
		 cpu_c0_n => cpu_c0_n, cpu_c1_n => cpu_c1_n,
		 cpu_skip_n => cpu_skip_n, IRQ => vIRQ,
		 kb_clk => kb_clk, kb_data => kb_data,
		 vga_RGB => vga_RGB,
		 vga_HS => vga_HS,	vga_VS => vga_VS);

-----------------------------------------------------------------------------------------
-- Port bits IOT
-- for now, just implement programmable I/O ports, 3x 12 bit

iobits_if: if PIOBASE /= O"00" generate

	signal bits_bank: unsigned(0 to 1);
	signal bits_sel: boolean;
	signal bits_dir: std_logic_vector(iobits'range);
	signal bits_out: std_logic_vector(iobits'range);

begin

	process (IOTact, IOTdev)
	begin
		bits_sel <= IOTact and (IOTdev = PIOBASE);
	end process;

	process (bits_out, bits_dir)
	begin
		for i in iobits'range loop
			if bits_dir(i) = '0' then
				iobits(i) <= 'Z';
			else
				iobits(i) <= bits_out(i);
			end if;
		end loop;
	end process;

	-- reading
	process (IOTread, bits_sel, bits_bank, iobits)
	begin
		if bits_sel and IOTread then
			case bits_bank is
				when "01" => dx <= iobits(12 to 23);
				when "10" => dx <= iobits(24 to 35);
				when others => dx <= iobits(0 to 11);
			end case;
		else
			dx <= (others => 'Z');
		end if;
	end process;

	-- writing
	process (reset, bits_sel, clk_write_n, dx)
	begin
		if reset then
			bits_dir <= (others => '0');	-- inputs on reset
		elsif rising_edge(clk_write_n) then
			if bits_sel then
				if IOTcmd = PIOSETBITS then -- write bits
					case bits_bank is
						when "01" => bits_out(12 to 23) <= dx;
						when "10" => bits_out(24 to 35) <= dx;
						when others => bits_out(0 to 11) <= dx;
					end case;
				elsif IOTcmd = PIOSETBANK then	-- select bank
					bits_bank <= unsigned(dx(10 to 11));
				elsif IOTcmd = PIOSETDIR then -- set direction
					case bits_bank is
 						when "01" => bits_dir(12 to 23) <= dx;
 						when "10" => bits_dir(24 to 35) <= dx;
 						when others => bits_dir(0 to 11) <= dx;
 					end case;
				end if;
			end if;
		end if;
	end process;

end generate; -- if PIOBASE /= O"00"

end RTL;
