-----------------------------------------------------------------------------------------
-- KL8JA serial port
-----------------------------------------------------------------------------------------

library IEEE;
use IEEE.STD_LOGIC_1164.ALL;
use IEEE.numeric_std.ALL;

use work.IOB_Config.ALL;

entity KL8JA is
	generic
	(
	print_addr: DevID := O"03";
	keyb_addr: DevID := O"04"
	);
	port
	( -- clocks
	clk: in std_logic;
	brg: in unsigned(7 downto 0);

	-- CPU bus interface
	reset : in boolean;
	IOTact: in boolean;
	IOTdev: in DevID;
	IOTcmd: in DevCmd;

	clk_write_n : in std_logic;
	clk_lxdar_n : in std_logic;
	IOTread : in boolean;
	dx : inout std_logic_vector(0 to 11);
	cpu_c0_n : out std_logic;
	cpu_c1_n : out std_logic;
	cpu_skip_n : out std_logic;
	IRQ: out std_logic;

	-- serial port
	txd : out std_logic;
	rxd : in std_logic
	);
end KL8JA;

architecture RTL of KL8JA is

	-- PosEdge: in this implementation, most processes are
	-- synchronous to clk, so this component is used to sample
	-- slower signals, and generates a 1-clock pulse on their rising edge
	component PosEdge
	port
	  (reset: in boolean;
		clk: in std_logic;
		inp: in std_logic;          
		outp: out std_logic);
	end component;

	-- common
	signal uiflag,														-- "keyboard flag"
			uoflag,														-- "printer flag"
			IE,															-- Interrupt Enable
			SE: std_logic;												-- Status Enable

	signal baud_sel: unsigned(0 to 2);							-- baud config register
				-- "000".."111" -> 38400, 19200, 9600, 4800, 1200, 600, 300, 150

		-- these next 3 registers are stored but not used yet
--	signal bits_sel: unsigned(0 to 1);							-- # of bits config register
--	signal parity_sel: unsigned(0 to 1);						-- parity config register
--	signal stops_sel: unsigned(0 to 0);							-- # of stop bits config register

	signal clk16_raw, clk16: std_logic;					-- clocking
	signal div16: unsigned(3 downto 0);

	-- receive part
	signal rsel, rclrflag: boolean;
	signal RBR,															-- receive buffer register
			RR: std_logic_vector(7 downto 0);					-- receive data register
	signal framing_error, overrun_error: std_logic;
	signal rbit: integer range 0 to 11;
		-- 0 = waiting
		-- 1 = start bit
		-- 2-10 = data bits
		-- 11 = stop bit
	signal inpv: std_logic_vector(0 to 2);
	signal startdet: boolean;
	signal samplet: unsigned(0 to 3);
	signal RBRF_1, RBRF_2: std_logic;							-- Receive Buffer Register Full

	-- transmit part
	signal tsel, write: boolean;
	signal tsetflag, tclrflag: boolean;
	signal sendt: unsigned(0 to 3);
	signal TRE: boolean;												-- Transmit Register Empty
	signal TR: std_logic_vector(0 to 8);						-- Transmit Register
	signal tbit: integer range 0 to 10;
		-- 0 = start bit
		--	1-8 = data bits
		-- 9 = stop bit
		-- 10 = ready
	signal THR: std_logic_vector(0 to 7);						-- Transmit Holding Register
	signal THRE_1, THRE_2, THRE: std_logic;	-- Transmit Holding Register Empty

begin

	-- clock select
	clk16_raw <= brg(to_integer(baud_sel));
	baud: PosEdge port map (reset, clk, clk16_raw, clk16);

	-- generate /16 clock
	process (reset, clk, clk16, div16)
	begin
		if reset then
			div16 <= (others => '0');
		elsif rising_edge(clk) then
			if clk16 = '1' then
				if div16 = 15 then
					div16 <= "0000";
				else
					div16 <= div16 + 1;
				end if;
			end if;
		end if;
	end process;

	rsel <= IOTact and (IOTdev = keyb_addr);
	IRQ <= IE and (uoflag or uiflag);

	tsel <= IOTact and (IOTdev = print_addr);
	write <= tsel and (clk_write_n = '0');

	-- manage the registers set by KIE
	-- the IOT is extended here for serial format and speed setting
	-- format:
	--     1 1 b b b n n p p s S I	set baud, number of bits, parity, stop bits, SE, IE
	--     X X X X X X X X X X S I	set SE and IE
	process (reset, clk_write_n)
	begin
		if reset then
			baud_sel <= "010";	-- 9600 baud
			IE <= '1';
			SE <= '0';
		elsif rising_edge(clk_write_n) then
			if rsel and (IOTcmd = KIE) then
				SE <= dx(10);
				IE <= dx(11);
				if (dx(0) = '1') and (dx(1) = '1') then
					baud_sel <= unsigned(dx(2 to 4));
					-- future use --
--					bits_sel <= unsigned(dx(5 to 6));
--					parity_sel <= unsigned(dx(7 to 8));
--					stops_sel <= unsigned(dx(9 to 9));
				end if;
			end if;
		end if;
	end process;

	-- c0/1/skip feedback
	process (rsel, iotcmd, uiflag)
	begin
		if rsel then
			-- c0/c1
			case IOTcmd is

				when KCC | KRB =>
					cpu_c0_n <= '0';

				when others =>
					cpu_c0_n <= 'Z';

			end case;

			case IOTcmd is

				when KRS | KRB =>
					cpu_c1_n <= '0';

				when others =>
					cpu_c1_n <= 'Z';

			end case;

			-- skip
			case IOTcmd is

				when KSF =>
					if uiflag = '1' then
						cpu_skip_n <= '0';
					else
						cpu_skip_n <= 'Z';
					end if;

				when others =>
					cpu_skip_n <= 'Z';

			end case;
		else
			cpu_c0_n <= 'Z';
			cpu_c1_n <= 'Z';
			cpu_skip_n <= 'Z';
		end if;
	end process;

	-- put data out to the CPU when requested
	process (rsel, IOTread, RBR, se, framing_error, overrun_error)
	begin
		if rsel and IOTread then
			dx(4 to 11) <= RBR;
			if SE = '1' then
				dx(0 to 3) <= (framing_error or overrun_error) &
						    '0' & framing_error & overrun_error;
			else
				dx(0 to 3) <= (others => '0');
			end if;
		else
			dx <= (others => 'Z');
		end if;
	end process;

	-- manage "keyboard flag"
	uiflag <= '1' when RBRF_1 /= RBRF_2 else '0';

	-- RBR is 'cleared' after IOT that resets flag
	process (rsel, clk_write_n, IOTcmd)
	begin
		if rising_edge(clk_write_n) then
			if rsel and ((IOTcmd = KCF) or (IOTcmd = KCC) or (IOTcmd = KRB)) then
				rclrflag <= true;
			else
				rclrflag <= false;
			end if;
		end if;
	end process;

	process (reset, clk_lxdar_n, rclrflag, RBRF_1)
	begin
		if reset then
			RBRF_2 <= '0';
		elsif rising_edge(clk_lxdar_n) then
			if rclrflag then
				RBRF_2 <= RBRF_1;
			end if;
		end if;
	end process;


	-- receive data
	---- 'vote' last three samples for start bit detection
	startdet <= (inpv = "000");

	process (reset, clk, clk16, RBRF_2)
	begin
		if reset then
			rbit <= 0;
			overrun_error <= '0';
			framing_error <= '0';
			RBRF_1 <= '0';
			inpv <= (others => '1');
			samplet <= (others => '0');
		elsif rising_edge(clk) then
			if clk16 = '1' then
				inpv <= (not rxd) & inpv(0 to 1);	-- keep last 3 samples
				if rbit = 0 then
					if startdet then
						rbit <= 1;						-- start receiving
						samplet <= div16 + "0111";	-- should be the middle of the baud period
					end if;
				elsif div16 = samplet then
					if rbit = 10 then					-- end of receive?
						
						-- set overrun previous char hasn't been cleared
						if uiflag = '1' then
							overrun_error <= '1';
						end if;

						-- transfer received byte into RBR and set flag
						RBRF_1 <= not RBRF_2;
						RBR <= RR;
						
						-- this should be a stop bit
						if inpv(0) /= '1' then
							framing_error <= '1';
						end if;
	
						rbit <= 0;							-- receiver ready for another
					else
						-- shift in serial data bit
						RR <= inpv(0) & RR(7 downto 1);
						rbit <= rbit + 1;
					end if;
				end if;
			end if;
		end if;
	end process;

	tsetflag <= write and (IOTcmd = TFL);
	tclrflag <= write and ((IOTcmd = TCF) or (IOTcmd = TLS));

	-- manage transmit buffer register
	process (reset, clk_write_n, tsel, THRE_2)
	begin
		if reset then
			THRE_1 <= '0';
		elsif rising_edge(clk_write_n) then
			if tsel and ((IOTcmd = TPC) or (IOTcmd = TLS)) then
				THR <= dx(4 to 11);
				THRE_1 <= not THRE_2;
			end if;
		end if;
	end process;

	-- "printer flag" management
	THRE <= '1' when THRE_1 = THRE_2 else '0';

	process (reset, THRE, tsetflag, tclrflag)
	begin
		if reset or tclrflag then
			uoflag <= '0';
		elsif tsetflag then
			uoflag <= '1';
		elsif rising_edge(THRE) then
			uoflag <= '1';
		end if;
	end process;

	-- c0/1/skip feedback
	process (tsel, iotcmd, uoflag, uiflag)
	begin
		if tsel then
			-- skip
			case IOTcmd is

				when TSF =>
					if uoflag = '1' then
						cpu_skip_n <= '0';
					else
						cpu_skip_n <= 'Z';
					end if;

				when TSK =>
					if (uoflag or uiflag) = '1' then
						cpu_skip_n <= '0';
					else
						cpu_skip_n <= 'Z';
					end if;

				when others =>
					cpu_skip_n <= 'Z';

			end case;
		else
			cpu_skip_n <= 'Z';
		end if;
	end process;

	-- serial output
	txd <= not TR(8);
	TRE <= (tbit = 10);

	process (reset, clk, clk16, THRE_1, THRE)
	begin
		if reset then

			TR(8) <= '1';
			tbit <= 10;
			THRE_2 <= '0';

		elsif rising_edge(clk) then

			if TRE then

				-- if buffered data, start it
				if THRE = '0' then
					tbit <= 0; 	-- thus TRE => '1'
					TR <= THR & '0';
					sendt <= div16 + "1111";
				end if;

			elsif (clk16 = '1') and (div16 = sendt) then

				-- THRE sets one bit time after start
				if tbit = 0 then
					THRE_2 <= THRE_1;
				end if;

				tbit <= tbit + 1;
				TR <= '1' & TR(0 to 7);

			end if;
		end if;
	end process;

end RTL;
