/*---------------------------------------------------------------------------

MKIOBROM

This program reads the .bin file for the extension ROM and the FPGA bitfile,
formats them, and outputs them as a text file to be downloaded using the
FL command of BTS6120 (rev 220 and later).

   The download image file has two sections.
   The first section is the executable portion; 32768 bytes are
   reserved for it, corresponding to up to 21845 12-bit words.
   It's a little more complicated than that, though, because the
   8-bit ROM is converted to words by the routine that handles the
   ramdisk, and this only works on 4KB fields.  A field in this format
   holds 21*256=2688 words, so the maximum code size is 8 times that,
   or 21504 words.  Note also that the extension ROM bootstrap only
   reads the first field's worth; code longer than this must read
   its remaining parts itself in its first field chunk.
   The second section is the FPGA bitfile data.  96K bytes are
   reserved for it.  It is basically just copied, since both it and
   the ROM are byte oriented.  The one wrinkle here is that the FPGA
   considers dx11 (d0) the most significant bit, but the file is stored
   'normally', so the bits must be flipped during the copy.

BUGS: not very parameterized

---------------------------------------------------------------------------*/

#include <stdio.h>
#include <stdlib.h>
#include <string.h>

/*-------------------------------------------------------------------------*/

const int BINFLD = 0;	/* starting field (4KB block) of code */
const int BITFLD = 8;	/* starting field (4KB block) of bitfile */
const int ENDFLD = 32;	/* starting field (4KB block) of unused ROM */

const unsigned CODEBASE = 030000;	/* code address base */

/*-------------------------------------------------------------------------*/

/*
   writeDL() - the output file is in conventional DL/RL/FL format with a
   record size of 256 bytes
*/
static FILE* out;
char outbuf[BUFSIZ];
static unsigned pos = 0;
static unsigned chk = 0;

void writeDL(unsigned char v)
{
	/* if at start of line, write address */
	if (!(pos & 7))
		{
		fprintf(out, "%04o.%03o/ ", pos / 256, pos & 255);
		}

        /* write data, update block checksum */
	fprintf(out, "%03o ", v);
	chk = (chk + v) & 07777;

        /* end the line after 8 bytes */
	++pos;
	if (!(pos & 7))
		fprintf(out, "\r\n");

        /* end the block after 256 bytes */
	if (!(pos & 255))
		{
		fprintf(out, "%04o\r\n", chk);
		chk = 0;
		}
}

/*
   flush() - make sure that the last block is complete
*/
void flush(void)
{
	while (pos & 255)
        	writeDL(0);
}

/*
   readBIN() - read the code file.  This is in DEC binload format
   and it not necessarily in order, so we read into a buffer, then
   convert that into a 2-into-3 format that the BTS6120 UNPACK
   function understands, and write it out.  This routine also
   blocks the output into fields to match the way PACK and UNPACK
   work, and inserts a checksum at reserved location 0200.
*/
void readBIN(FILE* in, unsigned max)
{
	unsigned short* code;
	unsigned maxaddr = 0, addr = 0200;
	unsigned n, m, chksum = 0;

	code = (unsigned short*) calloc(max, sizeof(unsigned short));

	/* skip header, if any */
	while (fgetc(in) != 0200) {}

	while (!feof(in))
		{
		/* read a frame */
		int c2, c1 = fgetc(in);
		switch (c1 & 0300)
			{
		case 0200:	/* leader/trailer */
			break;

		case 0300:	/* field */
			addr = (addr & 07777) | ((c1 & 0070) << 9);
			break;

		default:
			c2 = (c1 << 6) | fgetc(in);
			if ((c1 & 0300) == 0100)	/* offset */
				{
				addr = (addr & 070000) | (c2 & 07777);
				}
			else if (addr < max + CODEBASE)
				{
				if (addr < CODEBASE)
					{
					fprintf(stderr, "Fatal: BIN has code"
							" before field 2!\r\n");
					exit(7);
					}
				else
					{
					if (addr > maxaddr)
						maxaddr = addr;
					code[addr - CODEBASE] = c2;
					++addr;
					}
				}
			else
				{
				fprintf(stderr, "Fatal: BIN exceeds max size"
						" of %u words!\r\n", max);
				exit(5);
				}
			}
		}

	/* calculate a checksum for the initial ROM field and put it
	   in location 0200 */
	code[0200] = 0;
	for (m = 0; m < 2688; ++m)
		chksum += code[m];
	code[0200] = (~chksum + 1) & 07777;

	/* now emit it in 2-in-3 format, accounting for field boundaries.
           output in full fields so that checksum works */

	n = 0;
        for (;;)
		{
		/* put the word out in 2-into-3 format */
		writeDL(code[n+0]);
		writeDL(code[n+1]);
		writeDL( ((code[n+0] >> 8) & 0017) | ((code[n+1] >> 4) & 0360) );
                n += 2;

                if ((pos & 4095) == 4032)	/* last 12-bit word in field */
                	{
			while (pos & 4095)
                        	writeDL(0);
                        /* any more fields? */
                        if (n >= maxaddr - CODEBASE)
                        	break;
                        }
		}

	while (pos & 255)
		writeDL(0);

	printf("BIN written - %u ROM fields\r\n", ((maxaddr - CODEBASE) / 2688) + 1);
}

/*
   readBIT() - read the FPGA bitfile.  This is just a byte stream, and
   all that has to be done is to reverse the bits in each byte.
*/
void readBIT(FILE* in, unsigned max)
{
	unsigned n, total = 0;
	unsigned char buf[65536];
	unsigned char rev[256];

        /* build a bit-reversing array.  This is required because
           the FPGA expects the bitfile with b0 as msb. */
	for (n = 0; n < 256; ++n)
		{
		unsigned j = n;
		j = ((j & 0xAA) >> 1) | ((j & 0x55) << 1);
		j = ((j & 0xCC) >> 2) | ((j & 0x33) << 2);
		j = ((j & 0xF0) >> 4) | ((j & 0x0F) << 4);
		rev[n] = (unsigned char) j;
            	}

	while (!feof(in))
		{
		unsigned read = fread(buf, 1, sizeof(buf), in);

		if (total + read > max)
			{
			fprintf(stderr, "Fatal: BIT exceeds max size"
					" of %u bytes!\r\n", max);
			exit(6);
			}

		for (n = 0; n < read; ++n)
			writeDL(rev[buf[n]]);
		total += read;
		if (read < sizeof(buf))
			break;
		}

	printf("BIT written - %u ROM fields\r\n", (total / 4096) + 1);
}

/*-------------------------------------------------------------------------*/

int main(int argc, char* argv[])
{
	FILE *inbin = 0, *inbit = 0;

	if (argc != 4)
		{
		fprintf(stderr, "usage: MKDLROM <code file name> <FPGA file "
				"name> <output name>\r\n");
		fprintf(stderr, "  replace file name with '-' to omit\r\n");
		exit(1);
		}

	if (strcmp(argv[1], "-"))
		{
		inbin = fopen(argv[1], "rb");
		if (!inbin)
			{
			fprintf(stderr, "fatal: code binfile %s not found\r\n", argv[1]);
			exit(2);
			}
		}

	if (strcmp(argv[2], "-"))
		{
		inbit = fopen(argv[2], "rb");
		if (!inbit)
			{
			fclose(inbin);
			fprintf(stderr, "fatal: FPGA bitfile %s not found\r\n", argv[2]);
			exit(3);
			}
		}

	out = fopen(argv[3], "wb");
	if (!out)
		{
		fclose(inbin);
		fclose(inbit);
		fprintf(stderr, "fatal: output file %s not opened\r\n", argv[3]);
		exit(4);
		}

	setbuf(out, outbuf);

	if (inbin)
		{
		readBIN(inbin, BITFLD*2688);
		flush();
		}

	if (inbit)
		{
		pos = BITFLD * 4096;
		readBIT(inbit, (ENDFLD-BITFLD)*4096);
		flush();
		}

	fclose(inbin);
	fclose(inbit);

	return 0;
}

/*-------------------------------------------------------------------------*/
